import React, { Component } from 'react';
import readLastLines from 'read-last-lines';
import os from 'os';
import path from 'path';
import { History } from 'history';
import { withStyles } from '@material-ui/core/styles';
import electron, { shell, ipcRenderer } from 'electron';
import fillTemplate from 'es6-dynamic-template';
import 'react-circular-progressbar/dist/styles.css';

import { buildNumber } from 'constants/versions';
import routes from 'constants/routes';
import { SUPPORT_URL, RELEASE_NOTES_URLS } from 'constants/urls';
import HeaderLabel from 'components/HeaderLabel';
import { Logger } from 'library/Logger';
import {
    EVENTS,
    STATUS_DOWNLOADING_VERSION_INFO,
    STATUS_DOWNLOADING_FW,
    STATUS_INSTALLING_FW,
    VERSION_STABLE,
    VERSION_BETA,
    MODELS,
    HEADSET_STATUS
} from 'library/headset/consts';
import updateIcon from 'resources/images/update.svg';
import noInternetIcon from 'resources/images/no_connection.svg';
import badgeIcon from 'resources/images/checkmark.svg';
import { modelToImage, modelToName } from 'library/utils/ModelUtils';
import CheckForUpdates from './CheckForUpdates';
import DeviceUpToDateScreen from './DeviceUpToDateScreen';
import NoInternet from './NoInternet';
import NewVersionAvailable from './NewVersionAvailable';
import UpdatingFirmware from './UpdatingFirmware';
import ChooseVersionModal from './ChooseVersionModal';
import ConfirmQuitModal from './ConfirmQuitModal';
import styles from './styles';
import * as Authentication from '../../library/Authentication';

type Props = {
    device: object,
    language: object,
    strings: object,
    history: History,
    classes: {
        [key: string]: string
    }
};

class UpdateDeviceScreen extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            device: props.device,
            language: props.language,
            strings: props.strings,
            checkingForUpdates: true,
            noInternet: false,
            newVersionAvailable: false,
            updatingFirmware: false,
            latestFwVersion: null,
            fwUpdateProgress: 0,
            updatingFirmwareStatus: props.strings.update_device_updating_title,
            superUserFWUrls: [],
            additionalVersionsAvailable: false,
            hasBetaVersion: false,
            hasAdditionalVersion: false,
            fwInstallVersion: VERSION_STABLE,
            askToChooseVersion: false,
            problemUpdatingFw: false,
            stableVersion: null,
            betaVersion: null,
            betaBuildNumber: null,
            showDeviceUpToDateScreen: false,
            showQuitConfirm: false,
            chosenLanguage: null
        };
    }

    componentDidMount() {
        Logger.info('UpdateDeviceScreen: Mounted');

        const {
            device: { notSupported, model, connected, isDfu },
            updatingFirmware
        } = this.state;

        if (notSupported || model === MODELS.PRO1) {
            this.moveToUnsupportedDevice();
        } else if (!connected && !isDfu && !updatingFirmware) {
            Logger.info(
                'UpdateDeviceScreen - componentDidMount - calling moveToMainScreen',
                connected
            );
            this.moveToMainScreen();
        }

        ipcRenderer.on(
            `${EVENTS.IS_NEW_FIRMWARE}-response`,
            (event, response) => this.onNewFirmwareCheck(event, response)
        );
        ipcRenderer.on(
            `${EVENTS.FIRMWARE_UPDATE}-progress`,
            (event, response) => this.onFirmwareUpdateProgress(event, response)
        );
        ipcRenderer.on(
            `${EVENTS.FIRMWARE_UPDATE}-response`,
            (event, response) => this.onFirmwareUpdateFinish(event, response)
        );
        ipcRenderer.on(
            `${EVENTS.RECOVERY_FIRMWARE_UPDATE}-progress`,
            (event, response) => this.onFirmwareUpdateProgress(event, response)
        );
        ipcRenderer.on(
            `${EVENTS.RECOVERY_FIRMWARE_UPDATE}-response`,
            (event, response) => this.onFirmwareUpdateFinish(event, response)
        );

        ipcRenderer.on('windowApplicationClose', () =>
            this.onApplicationClose()
        );

        if (isDfu) {
            // DFU mode - do firmware recovery immediately
            this.startFirmwareRecovery();
        } else {
            this.checkForUpdates();
        }
    }

    componentDidUpdate(prevProps, prevState) {
        const { language, strings, device } = this.props;
        const { problemUpdatingFw } = this.state;

        Logger.info(
            'UpdateDeviceScreen - componentDidUpdate - ',
            prevState.device,
            device,
            prevState.problemUpdatingFw,
            problemUpdatingFw,
            prevState.updatingFirmware
        );

        if (prevState.language !== language) {
            // Getting current status message to update it with new localization
            const prevStatus = prevState.updatingFirmwareStatus;
            const [prevStringMessage] = Object.keys(prevState.strings).filter(
                key => prevState.strings[key] === prevStatus
            );
            const updatedState = { language, strings };
            if (prevStringMessage) {
                updatedState.updatingFirmwareStatus =
                    strings[prevStringMessage];
            }
            this.setState(updatedState);
        }
        if (
            (device.notSupported && !prevState.device.notSupported) ||
            device.model === MODELS.PRO1
        ) {
            // Device not supported
            Logger.info(
                'componentWillUpdate - Not updating device - unsupported device found - ',
                prevState.device,
                device
            );
            this.moveToUnsupportedDevice();
        }
        if (
            (!prevState.device.connected && device.connected) ||
            prevState.device.fwVersion !== device.fwVersion
        ) {
            this.setState({ device });
        }
        if (
            device.connected !== HEADSET_STATUS.HEADSET_CONNECTED ||
            (prevState.device.connected === HEADSET_STATUS.HEADSET_CONNECTED &&
                device.connected !== HEADSET_STATUS.HEADSET_CONNECTED &&
                !prevState.problemUpdatingFw &&
                !problemUpdatingFw &&
                !prevState.updatingFirmware)
        ) {
            // Device disconnected
            Logger.info(
                'UpdateDeviceScreen - componentDidUpdate - calling moveToMainScreen'
            );
            this.moveToMainScreen();
        }
    }

    componentWillUnmount() {
        ipcRenderer.removeAllListeners(`${EVENTS.IS_NEW_FIRMWARE}-response`);
        ipcRenderer.removeAllListeners(`${EVENTS.FIRMWARE_UPDATE}-progress`);
        ipcRenderer.removeAllListeners(`${EVENTS.FIRMWARE_UPDATE}-response`);
        ipcRenderer.removeAllListeners(
            `${EVENTS.RECOVERY_FIRMWARE_UPDATE}-progress`
        );
        ipcRenderer.removeAllListeners(
            `${EVENTS.RECOVERY_FIRMWARE_UPDATE}-response`
        );
        ipcRenderer.removeAllListeners('windowApplicationClose');
    }

    moveToUnsupportedDevice() {
        const { history } = this.props;
        const { device } = this.state;
        Logger.info('Not updating device - unsupported device found', device);

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(routes.UNSUPPORTED_UNIT.path);
        }, 100);
    }

    moveToMainScreen() {
        Logger.info('UpdateDeviceScreen - moveToMainScreen');
        const { history } = this.props;
        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            // Move to main screen, but do not perform auto update checks again
            history.replace({
                pathname: routes.WELCOME.path,
                state: { skipAutoUpdate: true }
            });
        }, 100);
    }

    moveToDeviceUpToDateScreen() {
        this.setState({ showDeviceUpToDateScreen: true });
    }

    moveToUnrecognizedDeviceScreen() {
        const { history } = this.props;
        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            // Move to main screen, but do not perform auto update checks again
            history.replace({
                pathname: routes.DEVICE_NOT_RECOGNIZED.path,
                state: { skipAutoUpdate: true }
            });
        }, 100);
    }

    moveToProblemUpdatingFirmwareScreen() {
        const { history } = this.props;
        const { device } = this.state;
        Logger.info('Not updating device - problem updating firmware', device);
        this.setState({ problemUpdatingFw: true });

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(routes.PROBLEM_UPDATING_FIRMWARE.path);
        }, 100);
    }

    moveToUpdateDeviceComplete() {
        const { history } = this.props;
        const { device } = this.state;
        Logger.info('Finished updating firmware', device);

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(routes.UPDATE_DEVICE_DONE.path);
        }, 100);
    }

    onApplicationClose() {
        const { updatingFirmware } = this.state;
        Logger.info(`Checking UpdateDevice on closing: ${updatingFirmware}`);
        if (updatingFirmware) {
            this.setState({ showQuitConfirm: true });
        } else {
            ipcRenderer.send('QUIT_CONFIRM');
        }
    }

    onFirmwareUpdateProgress(event, response) {
        Logger.info('onFirmwareUpdateProgress - ', response);
        const {
            strings,
            fwUpdateProgress,
            updatingFirmwareStatus
        } = this.state;

        let message = null;

        if (response.status === STATUS_DOWNLOADING_VERSION_INFO) {
            message = strings.update_device_status_downloading_version_info;
        } else if (response.status === STATUS_DOWNLOADING_FW) {
            message = strings.update_device_status_downloading_fw;
        } else if (response.status === STATUS_INSTALLING_FW) {
            message = strings.update_device_status_installing_fw;
        }

        if (typeof response.progress === 'number') {
            const newProgress = parseInt(response.progress, 10);
            if (newProgress !== fwUpdateProgress) {
                Logger.info(
                    'onFirmwareUpdateProgress - Setting progress',
                    newProgress
                );
                this.setState({
                    fwUpdateProgress: newProgress
                });
            }
        }

        if (message != null && message !== updatingFirmwareStatus) {
            this.setState({ updatingFirmwareStatus: message });
        }
    }

    onFirmwareUpdateFinish(event, response) {
        Logger.info('onFirmwareUpdateFinish - ', response);

        if (response.error) {
            // Move to error screen
            let error = null;
            if (typeof response.error === 'string') {
                error = response.error; // eslint-disable-line prefer-destructuring
            }
            Logger.info('onFirmwareUpdateFinish - Error', error);
            this.logFWEventDetails(error);

            this.moveToProblemUpdatingFirmwareScreen(error);
            return;
        }

        Logger.info('onFirmwareUpdateFinish - Finished');
        this.logFWEventDetails();
        this.moveToUpdateDeviceComplete();
    }

    // Compares if two FW versions - e.g. 5.10 vs 5.8 => 1; 5.10 vs 5.10 => 0; 5.8 vs 5.9 => -1.
    // Also supports patch version (e.g. 1.2.3 vs 1.2.4)
    compareFWVersions(v1, v2) {
        if (!v1 || !v2) return 0;

        const v1Parts = v1.split('.').map(x => parseInt(x, 10));
        const v2Parts = v2.split('.').map(x => parseInt(x, 10));

        const majorVersionDiff = v1Parts[0] - v2Parts[0];
        const minorVersionDiff = v1Parts[1] - v2Parts[1];
        const patchVersionDiff =
            v1Parts.length > 2 && v2Parts.length > 2
                ? v1Parts[2] - v2Parts[2]
                : 0;

        if (majorVersionDiff !== 0) {
            return majorVersionDiff;
        }
        if (minorVersionDiff !== 0) {
            return minorVersionDiff;
        }

        return patchVersionDiff;
    }

    // Returns an object with all details on the current system, update process, etc.
    logFWEventDetails(errorMessage) {
        const {
            fwUpdateProgress,
            device,
            fwInstallVersion,
            stableVersion
        } = this.state;
        // Get last few library log lines
        const logDir = electron.remote.app.getPath('userData');
        const libraryLogPath = path.join(logDir, 'cfu.log');

        if (stableVersion !== null) {
            Authentication.deviceFWUpdated(
                device,
                fwInstallVersion === VERSION_STABLE
                    ? stableVersion.toString()
                    : fwInstallVersion.toString(),
                device.fwVersion.toString(),
                fwUpdateProgress >= 100,
                fwUpdateProgress,
                errorMessage
            );
        }

        const details = {
            event:
                fwUpdateProgress < 100 ? 'fw_update_fail' : 'fw_update_success',
            os: process.platform,
            osVersion: os.release(),
            buildNumber,
            csrUpdateSuccess: fwUpdateProgress >= 50,
            tiUpdateSuccess: fwUpdateProgress === 100,
            error: fwUpdateProgress,
            lastCfuLibLogLines: null,
            isDfu: device.isDfu,
            modelSerial: device.serial,
            modelOriginalVersion: device.fwVersion,
            modelUpdatedVersion:
                fwInstallVersion === VERSION_STABLE
                    ? stableVersion
                    : fwInstallVersion
        };

        readLastLines
            .read(libraryLogPath, 50)
            .then(lines => {
                details.lastCfuLibLogLines = lines;
                Logger.info(JSON.stringify(details));
                return undefined;
            })
            .catch(error => {
                Logger.error(JSON.stringify(error));
            });
    }

    onNewFirmwareCheck(event, response) {
        const { device } = this.state;
        Logger.info('onNewFirmwareCheck - ', response);

        if (response.error) {
            if (!device.wasInterrupted) {
                this.setState({ checkingForUpdates: false, noInternet: true });
            } else {
                this.moveToUnrecognizedDeviceScreen();
            }
            return;
        }

        const newBTMBetaVersionAvailable =
            this.compareFWVersions(
                response.betaBuildNumber,
                device.btmVersion
            ) > 0;
        const newFWVersionAvailable =
            this.compareFWVersions(
                response.stableVersion,
                response.currentVersion
            ) > 0 ||
            (response.hasBeta &&
                this.compareFWVersions(
                    response.betaVersion,
                    response.currentVersion
                ) > 0) ||
            (response.super_user_urls &&
                response.super_user_urls.length > 0 &&
                this.compareFWVersions(
                    response.betaVersion,
                    response.currentVersion
                ) > 0 &&
                response.hasBeta) ||
            (newBTMBetaVersionAvailable && response.hasBeta);

        const additionalVersionsAvailable =
            response.hasBeta || response.super_user_urls.length > 0;

        this.setState({
            checkingForUpdates: false,
            newVersionAvailable: newFWVersionAvailable,
            latestFwVersion: response.stableVersion,
            betaVersion: response.betaVersion,
            betaBuildNumber: response.betaBuildNumber,
            additionalVersionsAvailable,
            hasBetaVersion: this.shouldShowBetaVersion(
                response.hasBeta,
                response.betaVersion,
                device.fwVersion,
                device.btmVersion,
                response.betaBuildNumber
            ),
            hasAdditionalVersion: response.super_user_urls.length > 0,
            superUserFWUrls: response.super_user_urls,
            stableVersion: response.stableVersion
        });

        if (!newFWVersionAvailable) {
            // Device is already up-to-date
            this.moveToDeviceUpToDateScreen();
        }
    }

    shouldShowBetaVersion(
        hasBeta,
        betaVersion,
        deviceFWVersion,
        deviceBTMVersion,
        betaBuildNumber
    ) {
        const showBetaVersion =
            hasBeta &&
            (this.compareFWVersions(betaVersion, deviceFWVersion) > 0 ||
                (!this.isDeviceBTMUpdated(betaBuildNumber, deviceBTMVersion) &&
                    betaVersion === deviceFWVersion));
        return showBetaVersion;
    }

    getBadgeIcon() {
        const {
            noInternet,
            newVersionAvailable,
            showDeviceUpToDateScreen
        } = this.state;
        if (noInternet) {
            return noInternetIcon;
        }
        if (newVersionAvailable) {
            return updateIcon;
        }
        if (showDeviceUpToDateScreen) {
            return badgeIcon;
        }
    }

    isDeviceBTMUpdated(betaBTM, deviceBTM) {
        const parsedBetaBTM = betaBTM.split('.');
        const parsedDeviceBTM = deviceBTM.split('.');
        if (
            parsedBetaBTM[0] > parsedDeviceBTM[0] ||
            parsedBetaBTM[1] > parsedDeviceBTM[1] ||
            parsedBetaBTM[2] > parsedDeviceBTM[2] ||
            (parsedBetaBTM[0] === parsedDeviceBTM[0] &&
                parsedBetaBTM[1] === parsedDeviceBTM[1] &&
                parsedBetaBTM[2] === parsedDeviceBTM[2])
        ) {
            return false;
        }
        return true;
    }

    onUpdateFirmware(lang) {
        const {
            additionalVersionsAvailable,
            device,
            hasBetaVersion,
            betaVersion,
            fwInstallVersion,
            superUserFWUrls,
            betaBuildNumber
        } = this.state;

        if (!additionalVersionsAvailable) {
            this.startInstallFirmware(lang);
            return;
        }

        // Let the user choose a firmware version to install
        Logger.info('Let user choose version');
        let updatedFwInstallVersion = fwInstallVersion;
        const [additionalVersionInstalled] = superUserFWUrls.filter(
            fwObject => fwObject.fw_version === device.fwVersion
        );
        if (
            hasBetaVersion &&
            device.fwVersion.toString() === betaVersion.toString() &&
            betaBuildNumber.toString() === device.btmVersion.toString()
        ) {
            updatedFwInstallVersion = VERSION_BETA;
        } else if (additionalVersionInstalled) {
            updatedFwInstallVersion = additionalVersionInstalled.fw_version;
        }
        this.setState({
            askToChooseVersion: true,
            chosenLanguage: lang,
            fwInstallVersion: updatedFwInstallVersion
        });
    }

    startFirmwareRecovery() {
        // Restart firmware installation with previously downloaded firmware
        this.setState({
            checkingForUpdates: false,
            updatingFirmware: true,
            newVersionAvailable: false
        });

        ipcRenderer.send(EVENTS.RECOVERY_FIRMWARE_UPDATE);
    }

    startInstallFirmware(lang) {
        const { fwInstallVersion } = this.state;
        this.setState({ updatingFirmware: true, newVersionAvailable: false });

        ipcRenderer.send(EVENTS.FIRMWARE_UPDATE, {
            version: fwInstallVersion,
            language: lang
        });
    }

    onNeedSupport() {
        shell.openExternal(SUPPORT_URL);
    }

    onWhatsNew() {
        const { device } = this.state;
        const url = RELEASE_NOTES_URLS[device.model];
        if (url) {
            shell.openExternal(url);
        }
    }

    checkForUpdates() {
        Logger.info('UpdateDeviceScreen: checkForUpdates');
        // Check for new FW version for this model
        this.setState({ checkingForUpdates: true, noInternet: false });
        Logger.info('UpdateDeviceScreen: sending IS_NEW_FIRMWARE');
        ipcRenderer.send(EVENTS.IS_NEW_FIRMWARE);
    }

    closeChooseVersion() {
        Logger.info('closeChooseVersion');
        this.setState({
            askToChooseVersion: false,
            fwInstallVersion: VERSION_STABLE
        });
    }

    onChooseVersion() {
        const { fwInstallVersion, chosenLanguage } = this.state;
        Logger.info('onChooseVersion', fwInstallVersion);
        this.setState({ askToChooseVersion: false });

        this.startInstallFirmware(chosenLanguage);
    }

    onCancel() {
        this.setState({ showQuitConfirm: false });
        ipcRenderer.send('CANCEL_CONFIRM');
    }

    onConfirm() {
        ipcRenderer.send('QUIT_CONFIRM');
        this.setState({ showQuitConfirm: false });
    }

    renderContent() {
        const {
            checkingForUpdates,
            noInternet,
            newVersionAvailable,
            updatingFirmware,
            showDeviceUpToDateScreen,
            language,
            hasBetaVersion,
            betaVersion,
            strings,
            device,
            updatingFirmwareStatus,
            fwUpdateProgress
        } = this.state;
        const { history } = this.props;

        if (!device.connected && !device.isDfu) {
            history.replace({
                pathname: routes.WELCOME.path,
                state: { skipAutoUpdate: true }
            });
            return;
        }
        if (checkingForUpdates) {
            return <CheckForUpdates strings={strings} />;
        }
        if (noInternet) {
            return (
                <NoInternet
                    strings={strings}
                    checkForUpdates={() => this.checkForUpdates()}
                    onNeedSupport={() => this.onNeedSupport()}
                    device={device}
                />
            );
        }
        if (newVersionAvailable) {
            return (
                <NewVersionAvailable
                    strings={strings}
                    device={device}
                    hasBetaVersion={hasBetaVersion}
                    betaVersion={betaVersion}
                    onWhatsNew={
                        RELEASE_NOTES_URLS[device?.model]
                            ? () => this.onWhatsNew()
                            : null
                    }
                    onUpdateFirmware={lang => this.onUpdateFirmware(lang)}
                />
            );
        }
        if (updatingFirmware) {
            Logger.info('renderContent - fwUpdateProgress ', fwUpdateProgress);
            return (
                <UpdatingFirmware
                    strings={strings}
                    device={device}
                    language={language}
                    updatingFirmwareStatus={updatingFirmwareStatus}
                    fwUpdateProgress={fwUpdateProgress}
                />
            );
        }
        if (showDeviceUpToDateScreen) {
            return (
                <DeviceUpToDateScreen
                    strings={strings}
                    device={device}
                    hasBetaVersion={hasBetaVersion}
                    betaVersion={betaVersion}
                    language={language}
                    onReInstallFirmware={lang => this.onUpdateFirmware(lang)}
                />
            );
        }
    }

    render() {
        const {
            device,
            strings,
            language,
            updatingFirmware,
            askToChooseVersion,
            latestFwVersion,
            fwInstallVersion,
            superUserFWUrls,
            checkingForUpdates,
            showQuitConfirm,
            hasBetaVersion,
            hasAdditionalVersion,
            betaVersion,
            betaBuildNumber
        } = this.state;
        const { classes } = this.props;
        return (
            <div className={classes.container}>
                {!device.isDfu && (
                    <HeaderLabel
                        strings={strings}
                        language={language}
                        showInfo={device}
                        hasBetaVersion={hasBetaVersion}
                        betaVersion={betaVersion}
                    >
                        {fillTemplate(strings.update_device_header, {
                            name: device.btName
                                ? device.btName
                                : modelToName(device.model)
                        })}
                    </HeaderLabel>
                )}
                <div className={classes.innerContainer}>
                    {!updatingFirmware && (
                        <div className={classes.imageContainer}>
                            <img
                                className={classes.mainImage}
                                src={modelToImage(device.model)}
                                alt="Device icon"
                            />
                            {!checkingForUpdates && (
                                <img
                                    className={classes.badgeImage}
                                    src={this.getBadgeIcon()}
                                    alt="Badge icon"
                                />
                            )}
                        </div>
                    )}
                    <div className={classes.contentContainer}>
                        {this.renderContent()}
                    </div>
                </div>

                <ChooseVersionModal
                    open={askToChooseVersion}
                    strings={strings}
                    fwInstallVersion={fwInstallVersion}
                    latestFwVersion={latestFwVersion}
                    betaVersion={betaVersion}
                    betaBuildNumber={betaBuildNumber}
                    superUserFWUrls={superUserFWUrls}
                    hasBetaVersion={hasBetaVersion}
                    hasAdditionalVersion={hasAdditionalVersion}
                    onClose={() => this.closeChooseVersion()}
                    onVersionChange={event =>
                        this.setState({ fwInstallVersion: event.target.value })
                    }
                    closeChooseVersion={() => this.closeChooseVersion()}
                    onChooseVersion={() => this.onChooseVersion()}
                />

                <ConfirmQuitModal
                    open={showQuitConfirm}
                    strings={strings}
                    onConfirm={() => this.onConfirm()}
                    onCancel={() => this.onCancel()}
                />
            </div>
        );
    }
}

export default withStyles(styles)(UpdateDeviceScreen);
